# metadata.py
import json
import os
import requests

TRACKS_FILENAME = "tracks.json"
COVERS_FOLDER = "covers"
PIKAPI_STUB = "https://www.radiofrance.fr/pikapi/images/"

class Track(dict):
    def __init__(self, firstLine, secondLine, uuid, cover, startTime, endTime):
        super().__init__(
            firstLine=firstLine, secondLine=secondLine,
            uuid=uuid, cover=cover,
            startTime=startTime, endTime=endTime
        )

    def save_cover(self):
        target_filename = f"{COVERS_FOLDER}/{self['cover']}.webp"
        if not os.path.exists(COVERS_FOLDER):
            os.makedirs(COVERS_FOLDER)
        if os.path.exists(target_filename):
            return
        r = requests.get(f"{PIKAPI_STUB}{self['cover']}/576x576")
        if r.status_code == 200:
            with open(target_filename, "wb") as f:
                f.write(r.content)
            print(f"[Metadata] Saved cover {target_filename}")
        else:
            print(f"[Metadata] Failed to fetch cover: {r.status_code}")

def read_tracks():
    if os.path.exists(TRACKS_FILENAME):
        with open(TRACKS_FILENAME, "r") as f:
            return json.load(f)
    return {}

def save_tracks(tracks):
    with open(TRACKS_FILENAME, "w") as f:
        json.dump(tracks, f)
    print(f"[Metadata] Saved {len(tracks)} tracks")

def add_track(tracks, track):
    uuid = track.get("secondLineSongUuid")
    if not uuid:
        print("[Metadata] Missing uuid, skipping track")
        return
    if uuid in tracks:
        return
    t = Track(
        track["firstLine"], track["secondLine"], uuid,
        track["cover"], track["startTime"], track["endTime"]
    )
    t.save_cover()
    tracks[uuid] = dict(t)

async def metadata_loop(tracks: dict):
    import asyncio
    url = "https://api.radiofrance.fr/livemeta/live/7/transistor_musical_player"
    while True:
        r = requests.get(url)
        if r.status_code == 200:
            data = r.json()
            for key in ("now", "prev", "next"):
                item = data.get(key)
                if isinstance(item, list):
                    item = item[0] if item else None
                if item:
                    add_track(tracks, item)
            save_tracks(tracks)
            delay = data.get("delayToRefresh", 10000) / 1000
        else:
            print(f"[Metadata] Error fetching: {r.status_code}")
            delay = 10
        await asyncio.sleep(delay)
