
#pragma once
#include <stdint.h>      // for uint8_t
#include <stdio.h>       // for size_t
#include <string.h>
#include "packet.h"

#define WIDTH   156

#define MAX_IDS     56
#define ID_LENGTH   21

#define OFFSETTIME  8
#define INFO_LEN    60

typedef uint64_t u64;

#define SPRINTF_MAC(straddr, addr) ( \
    snprintf(straddr, ID_LENGTH, \
    "%02hhX:%02hhX:%02hhX:%02hhX:%02hhX:%02hhX", \
    addr[5], addr[4], addr[3], addr[2], addr[1], addr[0]) )

#define SPRINTF_IPV4(straddr, addr) ( \
    snprintf(straddr, ID_LENGTH, \
    "%d.%d.%d.%d", \
    addr[0], addr[1], addr[2], addr[3]) )

#define MAC_TO_ID(ma)   ( ((u64)ma.addr[0] << 40) \
    | ((u64)ma.addr[1] << 32) \
    | ((u64)ma.addr[2] << 24) \
    | ((u64)ma.addr[3] << 16) \
    | ((u64)ma.addr[4] << 8) \
    | ((u64)ma.addr[5]) )

#define IPV4_TO_ID(ia)  ( ((u64)ia.addr[0] << 24) \
    | ((u64)ia.addr[1] << 16) \
    | ((u64)ia.addr[2] << 8) \
    | ((u64)ia.addr[3]) )

#define IPV6_TO_ID(ia)  ( ((u64)ia.addr[0] << 24) \
    | ((u64)ia.addr[1] << 16) \
    | ((u64)ia.addr[2] << 8) \
    | ((u64)ia.addr[3]) )

typedef long ID;

int found_ids = 0;

// stores everything relating to display of hosts on network
ID idmap[MAX_IDS];
char *idstrings[MAX_IDS];
int idwidth[MAX_IDS];

// stats counter
uint64_t total_packets = 0;
uint64_t total_bytes = 0;

double posix_time = 0;

char line[WIDTH];
char descr2[INFO_LEN];


int FILTER_L[8];

static int FIND_ID(ID id) {
    // if we find the id, return index
    for (int i = 0; i < found_ids; i++) {
        if (idmap[i] == id)
            return i;
    }

    // otherwise, add to list
    idmap[found_ids] = id;
    idstrings[found_ids] = (char*)malloc(sizeof(char) * ID_LENGTH);
    memset(idstrings[found_ids], ' ', ID_LENGTH);

    found_ids += 1;

    return MAX_IDS + found_ids - 1;
}

// draws arrows underneat packets
static void draw_print_line(int indexsrc, int indexdst, char *info) {
    int offsettime = 8;

    // find where the first bar should land
    int posindexsrc = (((indexsrc > 0) ? idwidth[indexsrc-1] : 0)) % (WIDTH - offsettime);
    // and the second
    int posindexdst = (((indexdst > 0) ? idwidth[indexdst-1] : 0)) % (WIDTH - offsettime);

    // find the actual starts and ends
    int reversed = (posindexdst < posindexsrc);

    int start = (reversed) ? posindexdst : posindexsrc;
    int end = (reversed) ? posindexsrc : posindexdst;

    int length = strlen(info);

    // start by writing how many packets we captured
    int wrote = 0;
    wrote = snprintf(line, offsettime, "%lu", total_packets);
    memset(line+wrote, ' ', WIDTH-wrote);

    // if print left -> right
    if (!reversed)
        snprintf(line+start+offsettime, WIDTH-start-offsettime-2, "| %s", info);
    else {
        // if it can fit on the left
        if (end - length - 2 >= 0) {
            snprintf(line+end+offsettime - length - 1, WIDTH - (end+offsettime - length), "%s", info);
            //strncat(line+end+offsettime - length - 2, info, WIDTH - end +offsettime);
            line[offsettime + end-1] = ' ';
            line[offsettime + end-0] = '|';
        }
        // otherwise print it on the right
        else
            snprintf(line+end+offsettime, WIDTH-end-offsettime+2, "| %s", info);
    }
    // output line
    printf("%-.*s\n", WIDTH, line);

    memset(line, ' ', WIDTH);
    wrote = snprintf(line, offsettime, "%.2fs", posix_time);
    line[wrote] = ' ';

    // draw the actual arrow
    memset(line+start+offsettime + 1, '_', (end - start - 1));
    line[(reversed ? end : start)+offsettime] = (reversed) ? '/' : '\\';
    line[(reversed ? start : end)+offsettime] = (reversed) ? '<' : '>';
    printf("%-.*s\n\n", WIDTH-1, line);
}

// prints list of identified network nodes
static void print_ids() {
    int count = 8;
    int indexnum = 0;

    int bufflen = (ID_LENGTH * (found_ids)) + 8;

    char* buff = malloc(bufflen * sizeof(char));
    int added = 8;

    // print border
    memset(buff, ' ', bufflen);

    memset(buff, '-', (bufflen < WIDTH) ? bufflen : WIDTH);
    printf("%.*s\n", WIDTH, buff);

    memset(buff, '\0', bufflen);

    char spot[ID_LENGTH];

    // repeatedly concatenate IDs to our buffer
    while (indexnum < found_ids) {
        //memset(spot, ' ', ID_LENGTH);
        int length = snprintf(spot, (int)ID_LENGTH, "| %.*s  ", ID_LENGTH - 4, idstrings[indexnum]);
        added += length;
        strncat(buff, spot, ID_LENGTH);

        // select n-1st
        idwidth[indexnum] = length + ((indexnum > 0) ? idwidth[indexnum-1] : 0);
        count = added;
        indexnum += 1;
    }

    int count2 = 0;

    // start actual printout
    memset(line, '\0', WIDTH);
    strcat(line, "| TIME  ");

    while (count2 < count) {
        printf("        %.*s\n", WIDTH-OFFSETTIME, buff+count2);
        // make sure we offset by each line's prefix
        count2 += WIDTH-OFFSETTIME;
    }

    // print closer
    memset(buff, '\0', bufflen);
    memset(buff, '-', (bufflen < WIDTH) ? bufflen : WIDTH);
    printf("%.*s\n", bufflen, buff);

    // release our ID buffer
    free(buff);
}

static void FLOW(void *tm_header, void *header, int layer, char *desc) {
    // if this layer should be filtered out
    if (FILTER_L[layer] == 1)
        return;

    memset(line, 0, WIDTH);

    ID idsrc;
    ID iddst;

    switch (layer) {
        case 2:
            idsrc = MAC_TO_ID(((MAC_HEADER *)tm_header)->src);
            iddst = MAC_TO_ID(((MAC_HEADER *)tm_header)->dst);
            break;
        default:
            idsrc = IPV4_TO_ID(((IPV4_HEADER *)tm_header)->src);
            iddst = IPV4_TO_ID(((IPV4_HEADER *)tm_header)->dst);
    }

    int indexsrc = FIND_ID(idsrc);

    // if ID didn't exist before
    if (indexsrc >= MAX_IDS) {
        // print out what it should be called
        if (layer == 2)
            SPRINTF_MAC(idstrings[indexsrc % MAX_IDS], ((MAC_HEADER*)tm_header)->src.addr);
        if (layer >= 3)
            SPRINTF_IPV4(idstrings[indexsrc % MAX_IDS], ((IPV4_HEADER*)tm_header)->src.addr);
    }

    int indexdst = FIND_ID(iddst);
    // if ID didn't exist before
    if (indexdst >= MAX_IDS) {
        // print out what it should be called
        if (layer == 2)
            SPRINTF_MAC(idstrings[indexdst % MAX_IDS], ((MAC_HEADER*)tm_header)->dst.addr);
        if (layer >= 3)
            SPRINTF_IPV4(idstrings[indexdst % MAX_IDS], ((IPV4_HEADER*)tm_header)->dst.addr);
    }
    // if either ID is new, print out header again
    if (indexsrc >= MAX_IDS || indexdst >= MAX_IDS)
         print_ids();

    // make sure we use the new indexes
    indexsrc %= MAX_IDS;
    indexdst %= MAX_IDS;

    // draw our arrow
    draw_print_line(indexsrc, indexdst, desc);

}

inline static char* SPRINTF_TCP(char *info, TCP_HEADER *th) {
    memset(info, '\0', INFO_LEN);

    int track = 0;
    track += snprintf(info, INFO_LEN, "TCP: %u --> %u ", th->srcprt, th->dstprt);

    if (ACK(th) | SYN(th)) {
        strncat(info, "[", 2);
        if (ACK(th))
            strncat(info, "ACK", 4);
        if (ACK(th) && SYN(th))
            strncat(info, ",", 2);
        if (SYN(th))
            strncat(info, "SYN", 4);
        strncat(info, "]", 2);
    }

    memset(descr2, '\0', INFO_LEN);

    // print format to our buffer
    snprintf(descr2, INFO_LEN, " Seq=%u Ack=%u Win=%u", th->seqnum, th->acknum, th->window);
    // add buffer to info line
    strncat(info, descr2, INFO_LEN - track);

    return info;
}
