#include <stdint.h>      // for uint8_t
#include <stdio.h>       // for size_t
#include <sys/socket.h>  // for recv, socket, AF_INET, SOCK_RAW
#include <arpa/inet.h>
#include <sys/types.h>
#include "byteswap.h"
#include <signal.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>

#include "packet.h"     // contains packet structs

#define FLOW_GRAPH  1

#include "flow_graph.h"

#define AF_PACKET       17
#define ETH_P_ALL       0x0003

uint8_t buff[65536];

clock_t start_t, end_t;
struct timespec start_tw, end_tw;

char *source = NULL;

char *descr;

// layer 4 UDP
void l4_udp(IPV4_HEADER *ih, uint8_t* data, size_t size) {
    _UDP_HEADER *_uh = (_UDP_HEADER *)data;
    UDP_HEADER *uh = BSWAP_UDP(_uh);

    snprintf(descr, INFO_LEN, "UDP: %d --> %d  Length: %d", uh->srcprt, uh->dstprt, uh->length);
    FLOW(ih, uh, 4, descr);
}

// layer 7 HTTP
void l7_http(IPV4_HEADER *ih, uint8_t *data, size_t size) {
    char* location2 = strstr((char *)data, "\r\n");

    if (location2 != 0) {
        snprintf(descr, INFO_LEN, "HTTP: %.*s size=%zu", (int)((unsigned char*)location2 - data), data, size);
        FLOW(ih, data, 7, descr);
    }
}

// layer 4 TCP
void l4_tcp(IPV4_HEADER *ih, uint8_t* data, size_t size) {
    _TCP_HEADER *_th = (_TCP_HEADER *)data;
    TCP_HEADER *th = BSWAP_TCP(_th);

    // if our TCP packet isn't empty
    if (size >= THL(th)) {
        switch (data[THL(th)]) {
            case 'H':   // HTTP
                l7_http(ih, data+THL(th), size - THL(th));
                break;
            case 0x17: {    // TLS
                snprintf(descr, INFO_LEN, "TLS: %u --> %u  Length=%zu", th->srcprt, th->dstprt, size - THL(th));
                FLOW(ih, data+THL(th), 7, descr);
                break;
            }
            default: {  // TCP Misc.
                SPRINTF_TCP(descr, th);
                FLOW(ih, th, 3, descr);
                break;
            }
        }
    }
}

// layer 3 IPv4
void l3_ipv4(MAC_HEADER *mh, uint8_t* data, size_t size) {
    // interpreter comme ipv4
    _IPV4_HEADER *_ih = (_IPV4_HEADER *)data;
    // effectuer byteswap
    IPV4_HEADER *ih = (IPV4_HEADER*)BSWAP_IPv4(_ih);

    //printf("sizeof IPV4: %ld\n", sizeof(IPV4_ADDR));
    //printf("IP header length: %d + %d = %ld\n", IHL(ih), ih->total_length - (IHL(ih)), size);

    // switch layer 3
    switch (ih->protocol) {
        case 0x06:  // TCP
            l4_tcp(ih, data+IHL(ih), size - IHL(ih));
            break;
        case 0x01:  // ICMP
            snprintf(descr, INFO_LEN, "ICMP: TTL = %d", ih->ttl);
            FLOW(ih, ih, 4, descr);
            break;
        case 0x11:  // UDP
            l4_udp(ih, data+IHL(ih), size - IHL(ih));
            break;
        default: {
            snprintf(descr, INFO_LEN, "Got layer 4 protocol = %04hhX", ih->protocol);
            FLOW(ih, ih, 4, descr);
        }
    }
}

void l3_ipv6(MAC_HEADER *mh, char* data, size_t size) {
        printf("IPv6:\t");

        printf("src:\t");
        for (size_t i = 0; i < 16; i++)
            printf((data[8+i] == 0) ? ":" : "%hhx:", (data[8+i]));

        printf("\tdst: ");
        for (size_t i = 0; i < 16; i++)
            printf ((data[24+i] == 0) ? ":" : "%hhx:", (data[24+i]));

        printf("\n");
}

// start packet decoding
void dump (uint8_t* data, size_t size) {
    // check when we are
    clock_gettime(CLOCK_MONOTONIC, &end_tw);
    double posix_wall = end_tw.tv_sec + 1e-9*end_tw.tv_nsec
                       - (start_tw.tv_sec + 1e-9*start_tw.tv_nsec);

    posix_time = posix_wall;

    memset(descr, 0, INFO_LEN);

    if (!FILTER_L[1] && !FLOW_GRAPH) {
        printf ("[%.2fs]\t[num %ld]\tfound %zu bytes\n", posix_wall, total_packets, size);
    }

    // check it's an ethernet frame
    if (size >= 14) {
        // Print mac header
        _MAC_HEADER *_mh = (_MAC_HEADER*)data;
        MAC_HEADER *mh = BSWAP_MAC(_mh);

        // switch layer 2
        switch (mh->type) {
            case 0x0800:    // IPv4
                //printf(" IPv4\n");
                l3_ipv4(mh, data+sizeof(MAC_HEADER), size-sizeof(MAC_HEADER));
                break;
            case 0x86dd:    // IPv6
                //printf(" IPv6!\n");
                //l3_ipv6(data+sizeof(MAC_HEADER), size-sizeof(MAC_HEADER));
                FLOW(mh, mh, 3, "IPv6!");
                break;
            case 0x0806:    // ARP
                FLOW(mh, mh, 2, (data+sizeof(MAC_HEADER))[7] == 1 ? "ARP Request" : "ARP Reply");
                break;
            default:        // OTHER L2
                FLOW(mh, mh, 2, "Got other layer 3");
                break;
        }
    }
}

// print stats on total bytes captured
static void onExit(int sig) {
    if (sig == 10)
        printf("ERROR: Packet size > 100KB. don't forget to setcap cap_net_raw the executable or run as sudo!\n");
    else
        printf("\nCollected %ld bytes in %ld packets!\n", total_bytes, total_packets);
    exit(0);
}

// find the file from filename
FILE* read_file(char *filename) {
    printf("Received file: \"%s\"\n", filename);
    FILE *input_file = fopen(filename, "r");
    if (input_file) {
        printf("Reading %s...\n", filename);
    }
    return input_file;
}

// parse filter argument list
void setup_filter(char *filter) {
    int user_choice = abs(atoi(filter));
    printf("user choice: %d\n", user_choice);
    if (user_choice != 0) {
        while (user_choice > 0) {
            int digit = user_choice  % 10;
            if (digit < 8)
                FILTER_L[digit] = 1;

            user_choice /= 10;
        }
    }
}

// main entrypoint
int main (int argc, char** argv) {
    int filemode = 1;

    descr = (char*)malloc((sizeof(char) * INFO_LEN + 2));

    // read from socket or file
    int filearg = 0;
    int filterarg = 0;
    FILE *input_file;

    if (argc == 3) {
        if (argv[1][0] == '-') {
            filearg = 2;
            filterarg = 1;
        }
        else if (argv[2][0] == '-') {
            filearg = 1;
            filterarg = 2;
        }
    }
    else if (argc == 2) {
        if (argv[1][0] == '-') {
            filemode = 0;
            filterarg = 1;
        }
        else {
            filearg = 1;
        }
    }
    else if (argc == 1) {
        filemode = 0;
    }
    if (filearg != 0) { {
        filemode = 1;
        input_file = read_file(argv[filearg]);
    }
    }
    if (filterarg != 0) {
        setup_filter(argv[filterarg]);
    }
#ifdef MACOS
    if (filemode == 0) {
        printf("ERROR: Live capture mode not supported on MacOS. You must specify an input file. See HOWTO.md for help.\n");
        onExit(1);
    }
#endif


    // attach to SIG_INT
    if (signal(SIGINT, onExit) == SIG_ERR)
        printf("Could not add signal handler...\n");

    // get program start time
    start_t = clock();
    clock_gettime(CLOCK_MONOTONIC, &start_tw);

    // if socket mode
    if (filemode == 0) {
        // create the socket
        int sockfd = socket (AF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
        if (!sockfd)
            exit(1);

        // receive and dump any data
        for (;;) {
            size_t size = recv (sockfd, buff, sizeof(buff), 0);
            if (size > 1000000)
                onExit(10);
            total_bytes += size;
            total_packets++;
            dump(buff, size);
        }
    }   // if reading from file
    else if (filemode == 1) {
        size_t newLen = 0;

        if (input_file != NULL) {
            // go to the end of the file
            if (fseek(input_file, 0L, SEEK_END) == 0) {
                // get the size of the file.
                long bufsize = ftell(input_file);
                if (bufsize == -1) { onExit(55); }

                // allocate our buffer to that size.
                source = malloc(sizeof(char) * (bufsize + 2));

                // go back to the start of the file.
                if (fseek(input_file, 0L, SEEK_SET) != 0) { exit(44); }

                // read the entire file into memory.
                newLen = fread(source, sizeof(char), bufsize, input_file);
                if ( ferror( input_file ) != 0 ) {
                    fputs("Error reading file", stderr);
                } else {
                    source[newLen++] = '\0'; // just to be safe
                }
            }
            // close up the file
            fclose(input_file);
        }

        // our tracking pointer
        char * new_source = source;

        // go through entire source file
        while (new_source < source + newLen) {
            char * location = strstr(new_source, "0000") + 7;
            char * location2 = strstr(new_source+1, "0000");

            // check that a next packet has been found
            if (location2 == 0 || location2 - location > sizeof buff)
                location2 = source + newLen;

            //printf("location of 1st \"%.10s\" location of second \"%.10s\"\n", location, location2);

            // go through char by char and either parse
            // as hex digits or whitespace

            int count = 0;
            for (; location < location2; ) {
                char ch0 = location[0];
                // check first character
                // if it's whitespace, skip
                if (ch0 == ' ')
                    location++;
                else if (ch0 == '\n')
                    location+=7;
                else if ( (ch0 <= 'f' && ch0 >= 'a')
                    || (ch0 <= 'F' && ch0 >= 'A')
                    || (ch0 <= '9' && ch0 >= '0') ) {
                    char ch1 = location[1];

                    // calculate actual byte values
                    uint8_t nib0 = ((ch0 & 0xF) + (ch0 >> 6)) | ((ch0 >> 3) & 0x8);
                    uint8_t nib1 = ((ch1 & 0xF) + (ch1 >> 6)) | ((ch1 >> 3) & 0x8);
                    buff[count] = (nib0 << 4) | nib1;
                    location+=2;
                    count++;
                }
                else {
                    location++;
                }
            }
            // add to our counters
            total_bytes += count;
            total_packets++;

            // dump packet
            dump(buff, count);
            new_source = location2;
        }

        free(source);
    }

    onExit(1);
}





















